/*
 * Demo ET-ESP32-RS485 Hardware Board
 * MCU      : ESP32-WROVER
 *          : Arduino Tools Board : ESP32 Wrover Module
 * Software : https://dl.espressif.com/dl/package_esp32_index.json        
 * Reserve  : Boot Config       
 *          : -> IO0(BOOT)
 *          : -> IO2(Don't Card on 3V3V Version)
 *          : -> IO5
 *          : -> IO12(MTDI)
 *          : -> IO15(MTDO)
 * Reserve  : SFLASH        
 *          : -> CMD
 *          : -> CLK
 *          : -> SD0
 *          : -> SD1
 *          : -> SD2
 *          : -> SD3
 * Debug    : Serial0 : USART0(USB)        
 *          : -> RX0(Debug & Download:IO3)
 *          : -> TX0(Debug & Download:IO1)
 * NB-IoT   : Serial1 : SIM7020E(BK-7020 V2)       
 *          : -> RX1(T:IO14)
 *          : -> TX1(R:IO13)
 *          : -> PWRKEY(K:IO33)
 *          : -> SLEEP(S:IO32)
 * RS485    : Serial1 : RS485  
 *          : -> RX2(IO26)
 *          : -> TX2(IO27)
 *          : -> DIR(Direction : IO25 : LOW = Receive, HIGH = Send)
 * I2C#1    : I2C BUS
 *          : -> SCL1(IO22)
 *          : -> SDA1(IO21)
 * RTC      : -> RTC:DS3231/DS3232
 *          : -> PCF8574/A(Relay8)
 *          : -> PCF8574/A(DC-IN8)
 * LED      : LED Status(Active High)
 *          : -> LED0(IO2)
 * Demo     : SD Card Interface         
 */

//=================================================================================================
#include "FS.h"
#include "SD.h"
#include "SPI.h" 
//=================================================================================================
#include <Wire.h> 
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP32-RS485
//=================================================================================================
// Remap Pin USART -> C:\Users\Admin\Documents\Arduino\hardware\espressif\esp32\cores\esp32\HardwareSerial.cpp
//                    C:\Users\Admin\AppData\Local\Arduino15\packages\esp32\hardware\esp32\1.0.0\cores\esp32\HardwareSerial.cpp
//=================================================================================================
#include <HardwareSerial.h>
//=================================================================================================
#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
#define SerialNBIOT_RX_PIN    14
#define SerialNBIOT_TX_PIN    13
#define SerialNBIOT  Serial1                                                                      // Serial1(IO13=TXD,IO14=RXD)
//=================================================================================================
#define SerialRS485_RX_PIN    26
#define SerialRS485_TX_PIN    27
#define SerialRS485  Serial2                                                                      // Serial2(IO27=TXD,IO26=RXD)
//=================================================================================================
#define RS485_DIRECTION_PIN   25                                                                  // ESP32-WROVER :IO25
#define RS485_RXD_SELECT      LOW
#define RS485_TXD_SELECT      HIGH
//=================================================================================================
#define SIM7020E_PWRKEY_PIN   33                                                                  // ESP32-WROVER :IO33
#define SIM7020E_SLEEP_PIN    32                                                                  // ESP32-WROVER :IO32
#define SIM7020E_PWRKEY_LOW   LOW                                                                 // Start Power-ON
#define SIM7020E_PWRKEY_HIGH  HIGH                                                                // Release Signal
#define SIM7020E_SLEEP_LOW    LOW                                                                 // Pull-Up DTR to Enable Sleep
#define SIM7020E_SLEEP_HIGH   HIGH                                                                // DTR=Low(Wakeup From Sleep)
//=================================================================================================
#define I2C_SCL1_PIN          22                                                                  // ESP32-WROVER : IO22(SCL1)
#define I2C_SDA1_PIN          21                                                                  // ESP32-WROVER : IO21(SDA1)
//=================================================================================================

//=================================================================================================
#define LED_PIN               2                                                                   // ESP-WROVER  : IO2
#define LedON                 1
#define LedOFF                0
//=================================================================================================
#define CS_SD_CARD_PIN        4                                                                   // ESP-WROVER  : IO4
#define SD_CARD_DISABLE       1
#define SD_CARD_ENABLE        0
//=================================================================================================
// End of Default Hardware : ET-ESP32-RS485
//=================================================================================================

void listDir(fs::FS &fs, const char * dirname, uint8_t levels)
{
  Serial.printf("Listing directory: %s\n", dirname);

  File root = fs.open(dirname);
  if(!root)
  {
    Serial.println("Failed to open directory");
    return;
  }
  if(!root.isDirectory())
  {
    Serial.println("Not a directory");
    return;
  }

  File file = root.openNextFile();
  while(file)
  {
    if(file.isDirectory())
    {
      Serial.print("  DIR : ");
      Serial.println(file.name());
      if(levels)
      {
        listDir(fs, file.name(), levels -1);
      }
    } 
    else 
    {
      Serial.print("  FILE: ");
      Serial.print(file.name());
      Serial.print("  SIZE: ");
      Serial.println(file.size());
    }
    file = root.openNextFile();
  }
}

void createDir(fs::FS &fs, const char * path)
{
  Serial.printf("Creating Dir: %s\n", path);
  if(fs.mkdir(path))
  {
    Serial.println("Dir created");
  } 
  else 
  {
    Serial.println("mkdir failed");
  }
}

void removeDir(fs::FS &fs, const char * path)
{
  Serial.printf("Removing Dir: %s\n", path);
  if(fs.rmdir(path))
  {
    Serial.println("Dir removed");
  } 
  else 
  {
    Serial.println("rmdir failed");
  }
}

void readFile(fs::FS &fs, const char * path)
{
  Serial.printf("Reading file: %s\n", path);

  File file = fs.open(path);
  if(!file)
  {
    Serial.println("Failed to open file for reading");
    return;
  }

  Serial.print("Read from file: ");
  while(file.available())
  {
    Serial.write(file.read());
  }
}

void writeFile(fs::FS &fs, const char * path, const char * message)
{
  Serial.printf("Writing file: %s\n", path);

  File file = fs.open(path, FILE_WRITE);
  if(!file)
  {
    Serial.println("Failed to open file for writing");
    return;
  }
  if(file.print(message))
  {
    Serial.println("File written");
  } 
  else 
  {
    Serial.println("Write failed");
  }
}

void appendFile(fs::FS &fs, const char * path, const char * message)
{
  Serial.printf("Appending to file: %s\n", path);

  File file = fs.open(path, FILE_APPEND);
  if(!file)
  {
    Serial.println("Failed to open file for appending");
    return;
  }
  if(file.print(message))
  {
    Serial.println("Message appended");
  } 
  else 
  {
    Serial.println("Append failed");
  }
}

void renameFile(fs::FS &fs, const char * path1, const char * path2)
{
  Serial.printf("Renaming file %s to %s\n", path1, path2);
  if(fs.rename(path1, path2)) 
  {
    Serial.println("File renamed");
  } 
  else 
  {
    Serial.println("Rename failed");
  }
}

void deleteFile(fs::FS &fs, const char * path)
{
  Serial.printf("Deleting file: %s\n", path);
  if(fs.remove(path))
  {
    Serial.println("File deleted");
  } 
  else 
  {
    Serial.println("Delete failed");
  }
}

void testFileIO(fs::FS &fs, const char * path)
{
  File file = fs.open(path);
  static uint8_t buf[512];
  size_t len = 0;
  uint32_t start = millis();
  uint32_t end = start;
  if(file)
  {
    len = file.size();
    size_t flen = len;
    start = millis();
    while(len)
    {
      size_t toRead = len;
      if(toRead > 512)
      {
        toRead = 512;
      }
      file.read(buf, toRead);
      len -= toRead;
    }
    end = millis() - start;
    Serial.printf("%u bytes read for %u ms\n", flen, end);
    file.close();
  } 
  else 
  {
    Serial.println("Failed to open file for reading");
  }

  file = fs.open(path, FILE_WRITE);
  if(!file)
  {
    Serial.println("Failed to open file for writing");
    return;
  }

  size_t i;
  start = millis();
  for(i=0; i<2048; i++)
  {
    file.write(buf, 512);
  }
  end = millis() - start;
  Serial.printf("%u bytes written for %u ms\n", 2048 * 512, end);
  file.close();
}

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP32-RS485
  //===============================================================================================
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LedOFF);
  //===============================================================================================
  pinMode(CS_SD_CARD_PIN, OUTPUT);
  digitalWrite(CS_SD_CARD_PIN, SD_CARD_DISABLE);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                                      
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);                                                          // RS485 Direction
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);
  //===============================================================================================
  SerialNBIOT.begin(115200, SERIAL_8N1, SerialNBIOT_RX_PIN, SerialNBIOT_TX_PIN);
  while(!SerialNBIOT);
  //===============================================================================================
  SerialRS485.begin(115200, SERIAL_8N1, SerialRS485_RX_PIN, SerialRS485_TX_PIN);
  while(!SerialRS485);
  //===============================================================================================
  // End of Initial Default Hardware : ESP32
  //===============================================================================================

  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo Interface SD Card");
  //===============================================================================================
}

void loop() 
{
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("SD Card Test...Start");
  //===============================================================================================
  if(!SD.begin())
  {
    //=============================================================================================
    SerialDebug.println("Card Mount Failed");
    //=============================================================================================
    return;
  }
  uint8_t cardType = SD.cardType();

  if(cardType == CARD_NONE)
  {
    //=============================================================================================
    SerialDebug.println("No SD card attached");
    //=============================================================================================
    return;
  }
  //===============================================================================================
  SerialDebug.print("SD Card Type: ");
  //===============================================================================================
  if(cardType == CARD_MMC)
  {
    //=============================================================================================
    SerialDebug.println("MMC");
    //=============================================================================================
  } 
  else if(cardType == CARD_SD)
  {
    //=============================================================================================
    SerialDebug.println("SDSC");
    //=============================================================================================
  } 
  else if(cardType == CARD_SDHC)
  {
    //=============================================================================================
    SerialDebug.println("SDHC");
    //=============================================================================================
  } 
  else 
  {
    //=============================================================================================
    SerialDebug.println("UNKNOWN");
    //=============================================================================================
  }

  uint64_t cardSize = SD.cardSize() / (1024 * 1024);
  //===============================================================================================
  SerialDebug.printf("SD Card Size: %lluMB\n", cardSize);
  //===============================================================================================
  listDir(SD, "/", 0);
  createDir(SD, "/mydir");
  listDir(SD, "/", 0);
  removeDir(SD, "/mydir");
  listDir(SD, "/", 2);
  writeFile(SD, "/hello.txt", "Hello ");
  appendFile(SD, "/hello.txt", "World!\n");
  readFile(SD, "/hello.txt");
  deleteFile(SD, "/foo.txt");
  renameFile(SD, "/hello.txt", "/foo.txt");
  readFile(SD, "/foo.txt");
  testFileIO(SD, "/test.txt");
  //===============================================================================================
  SerialDebug.println("Test SD Card...Complete");
  //===============================================================================================
  
  while(1);  
}

